-- Roles enum and table
DO $$ BEGIN
  CREATE TYPE public.app_role AS ENUM ('admin','moderator','user');
EXCEPTION
  WHEN duplicate_object THEN null;
END $$;

CREATE TABLE IF NOT EXISTS public.user_roles (
  id uuid PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id uuid REFERENCES auth.users(id) ON DELETE CASCADE NOT NULL,
  role public.app_role NOT NULL,
  UNIQUE (user_id, role)
);

ALTER TABLE public.user_roles ENABLE ROW LEVEL SECURITY;

-- Security definer function to check roles
CREATE OR REPLACE FUNCTION public.has_role(_user_id uuid, _role public.app_role)
RETURNS boolean
LANGUAGE sql
STABLE
SECURITY DEFINER
SET search_path = public
AS $$
  SELECT EXISTS (
    SELECT 1 FROM public.user_roles
    WHERE user_id = _user_id AND role = _role
  );
$$;

-- Programming schedule table
CREATE TABLE IF NOT EXISTS public.programming_schedule (
  id uuid PRIMARY KEY DEFAULT gen_random_uuid(),
  title text NOT NULL,
  host text,
  day_of_week text NOT NULL,
  start_time time NOT NULL,
  end_time time NOT NULL,
  description text,
  active boolean NOT NULL DEFAULT true,
  position int GENERATED BY DEFAULT AS IDENTITY,
  created_at timestamptz NOT NULL DEFAULT now(),
  updated_at timestamptz NOT NULL DEFAULT now()
);

ALTER TABLE public.programming_schedule ENABLE ROW LEVEL SECURITY;

-- update_updated_at helper
CREATE OR REPLACE FUNCTION public.update_updated_at_column()
RETURNS trigger AS $$
BEGIN
  new.updated_at = now();
  RETURN new;
END;
$$ LANGUAGE plpgsql SET search_path = public;

DROP TRIGGER IF EXISTS trg_programming_schedule_updated_at ON public.programming_schedule;
CREATE TRIGGER trg_programming_schedule_updated_at
BEFORE UPDATE ON public.programming_schedule
FOR EACH ROW EXECUTE FUNCTION public.update_updated_at_column();

-- Policies for programming_schedule
DROP POLICY IF EXISTS "Anyone can view schedule" ON public.programming_schedule;
CREATE POLICY "Anyone can view schedule"
  ON public.programming_schedule
  FOR SELECT
  USING (true);

DROP POLICY IF EXISTS "Only admins can insert schedule" ON public.programming_schedule;
CREATE POLICY "Only admins can insert schedule"
  ON public.programming_schedule
  FOR INSERT
  TO authenticated
  WITH CHECK (public.has_role(auth.uid(), 'admin'));

DROP POLICY IF EXISTS "Only admins can update schedule" ON public.programming_schedule;
CREATE POLICY "Only admins can update schedule"
  ON public.programming_schedule
  FOR UPDATE
  TO authenticated
  USING (public.has_role(auth.uid(), 'admin'));

DROP POLICY IF EXISTS "Only admins can delete schedule" ON public.programming_schedule;
CREATE POLICY "Only admins can delete schedule"
  ON public.programming_schedule
  FOR DELETE
  TO authenticated
  USING (public.has_role(auth.uid(), 'admin'));

-- Seed admin role for the known admin email
INSERT INTO public.user_roles (user_id, role)
SELECT u.id, 'admin'::public.app_role
FROM auth.users u
WHERE u.email = 'admin@radiomegagospel.com.br'
ON CONFLICT DO NOTHING;