import React, { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import { LogOut, Save } from "lucide-react";
import ScheduleManager from "@/components/admin/ScheduleManager";
import BannersManager from "@/components/admin/BannersManager";

const Admin = () => {
  const navigate = useNavigate();
  const { toast } = useToast();
  const [user, setUser] = useState(null);
  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [settings, setSettings] = useState({
    current_show: "",
    current_host: "",
    site_phone: "",
    site_email: "",
    whatsapp_number: "",
    radio_stream_url: "",
  });

  useEffect(() => {
    supabase.auth.getSession().then(({ data: { session } }) => {
      setUser(session?.user ?? null);
      if (session?.user) loadSettings();
    });

    const { data: { subscription } } = supabase.auth.onAuthStateChange((_, session) => {
      setUser(session?.user ?? null);
      if (session?.user) loadSettings();
    });

    return () => subscription.unsubscribe();
  }, []);

  const loadSettings = async () => {
    const { data } = await supabase.from("site_settings").select("*");
    if (data) {
      const settingsObj = data.reduce((acc, item) => ({ ...acc, [item.key]: item.value }), {} as Record<string, string>);
      setSettings({
        current_show: settingsObj.current_show || "",
        current_host: settingsObj.current_host || "",
        site_phone: settingsObj.site_phone || "",
        site_email: settingsObj.site_email || "",
        whatsapp_number: settingsObj.whatsapp_number || "",
        radio_stream_url: settingsObj.radio_stream_url || "",
      });
    }
  };

  const handleLogin = async (e: React.FormEvent) => {
    e.preventDefault();
    const { error } = await supabase.auth.signInWithPassword({ email, password });
    if (error) {
      toast({ title: "Erro no login", description: error.message, variant: "destructive" });
    } else {
      toast({ title: "Login realizado com sucesso!" });
    }
  };

  const handleSave = async () => {
    try {
      for (const [key, value] of Object.entries(settings)) {
        const { error } = await supabase.from("site_settings").upsert({ key, value }, { onConflict: "key" });
        if (error) throw error;
      }
      toast({ title: "Configurações salvas com sucesso!" });
    } catch (error) {
      toast({ title: "Erro ao salvar", description: "Tente novamente", variant: "destructive" });
    }
  };

  const handleLogout = async () => {
    await supabase.auth.signOut();
    toast({ title: "Logout realizado" });
    navigate("/");
  };

  if (!user) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-primary via-[hsl(var(--primary-variant))] to-primary p-4">
        <Card className="w-full max-w-md p-8">
          <h1 className="text-3xl font-bold text-center mb-6 bg-gradient-to-r from-primary to-[hsl(var(--primary-variant))] bg-clip-text text-transparent">
            Admin Login
          </h1>
          <form onSubmit={handleLogin} className="space-y-4">
            <Input type="email" placeholder="Email" value={email} onChange={(e) => setEmail(e.target.value)} required />
            <Input type="password" placeholder="Senha" value={password} onChange={(e) => setPassword(e.target.value)} required />
            <Button type="submit" variant="hero" className="w-full">Entrar</Button>
          </form>
        </Card>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background p-8">
      <div className="max-w-6xl mx-auto">
        <div className="flex justify-between items-center mb-8">
          <h1 className="text-4xl font-bold bg-gradient-to-r from-primary to-[hsl(var(--primary-variant))] bg-clip-text text-transparent">
            Painel Administrativo
          </h1>
          <Button variant="outline" onClick={handleLogout}><LogOut className="w-4 h-4" /> Sair</Button>
        </div>

        <Tabs defaultValue="settings" className="w-full">
          <TabsList className="grid w-full grid-cols-3">
            <TabsTrigger value="settings">Configurações</TabsTrigger>
            <TabsTrigger value="schedule">Programação</TabsTrigger>
            <TabsTrigger value="banners">Banners</TabsTrigger>
          </TabsList>

          <TabsContent value="settings">
            <Card className="p-8 space-y-6">
              <div>
                <label className="block text-sm font-medium mb-2">Programa Atual</label>
                <Input value={settings.current_show} onChange={(e) => setSettings({ ...settings, current_show: e.target.value })} />
              </div>
              <div>
                <label className="block text-sm font-medium mb-2">Locutor/Host</label>
                <Input value={settings.current_host} onChange={(e) => setSettings({ ...settings, current_host: e.target.value })} />
              </div>
              <div>
                <label className="block text-sm font-medium mb-2">Telefone do Site</label>
                <Input value={settings.site_phone} onChange={(e) => setSettings({ ...settings, site_phone: e.target.value })} />
              </div>
              <div>
                <label className="block text-sm font-medium mb-2">Email do Site</label>
                <Input value={settings.site_email} onChange={(e) => setSettings({ ...settings, site_email: e.target.value })} />
              </div>
              <div>
                <label className="block text-sm font-medium mb-2">Número WhatsApp</label>
                <Input value={settings.whatsapp_number} onChange={(e) => setSettings({ ...settings, whatsapp_number: e.target.value })} />
              </div>
              <div>
                <label className="block text-sm font-medium mb-2">URL do Stream de Rádio</label>
                <Input 
                  value={settings.radio_stream_url} 
                  onChange={(e) => setSettings({ ...settings, radio_stream_url: e.target.value })} 
                  placeholder="https://stm.painelmegahost.com.br:7004/stream"
                />
              </div>
              <Button variant="hero" className="w-full" onClick={handleSave}><Save className="w-4 h-4" /> Salvar Alterações</Button>
            </Card>
          </TabsContent>

          <TabsContent value="schedule">
            <ScheduleManager />
          </TabsContent>

          <TabsContent value="banners">
            <BannersManager />
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
};

export default Admin;
