import React, { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card } from "@/components/ui/card";
import { useToast } from "@/hooks/use-toast";
import { Trash2, Edit2, Plus, Save, X } from "lucide-react";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";

interface Schedule {
  id: string;
  title: string;
  host: string | null;
  day_of_week: string;
  start_time: string;
  end_time: string;
  description: string | null;
  active: boolean;
}

const DAYS_OF_WEEK = [
  "Segunda-feira",
  "Terça-feira",
  "Quarta-feira",
  "Quinta-feira",
  "Sexta-feira",
  "Sábado",
  "Domingo",
];

const ScheduleManager = () => {
  const { toast } = useToast();
  const [schedules, setSchedules] = useState<Schedule[]>([]);
  const [editingId, setEditingId] = useState<string | null>(null);
  const [isAdding, setIsAdding] = useState(false);
  const [formData, setFormData] = useState({
    title: "",
    host: "",
    day_of_week: "",
    start_time: "",
    end_time: "",
    description: "",
    active: true,
  });

  useEffect(() => {
    loadSchedules();
  }, []);

  const loadSchedules = async () => {
    const { data, error } = await supabase
      .from("programming_schedule")
      .select("*")
      .order("day_of_week")
      .order("start_time");

    if (error) {
      toast({ title: "Erro ao carregar grade", variant: "destructive" });
      return;
    }
    setSchedules(data || []);
  };

  const handleSave = async () => {
    if (!formData.title || !formData.day_of_week || !formData.start_time || !formData.end_time) {
      toast({ title: "Preencha os campos obrigatórios", variant: "destructive" });
      return;
    }

    const payload = {
      title: formData.title,
      host: formData.host || null,
      day_of_week: formData.day_of_week,
      start_time: formData.start_time,
      end_time: formData.end_time,
      description: formData.description || null,
      active: formData.active,
    };

    if (editingId) {
      const { error } = await supabase
        .from("programming_schedule")
        .update(payload)
        .eq("id", editingId);

      if (error) {
        toast({ title: "Erro ao atualizar", variant: "destructive" });
        return;
      }
      toast({ title: "Programa atualizado!" });
    } else {
      const { error } = await supabase.from("programming_schedule").insert(payload);

      if (error) {
        toast({ title: "Erro ao criar", variant: "destructive" });
        return;
      }
      toast({ title: "Programa criado!" });
    }

    resetForm();
    loadSchedules();
  };

  const handleEdit = (schedule: Schedule) => {
    setEditingId(schedule.id);
    setFormData({
      title: schedule.title,
      host: schedule.host || "",
      day_of_week: schedule.day_of_week,
      start_time: schedule.start_time,
      end_time: schedule.end_time,
      description: schedule.description || "",
      active: schedule.active,
    });
    setIsAdding(true);
  };

  const handleDelete = async (id: string) => {
    if (!confirm("Tem certeza que deseja excluir este programa?")) return;

    const { error } = await supabase.from("programming_schedule").delete().eq("id", id);

    if (error) {
      toast({ title: "Erro ao excluir", variant: "destructive" });
      return;
    }

    toast({ title: "Programa excluído!" });
    loadSchedules();
  };

  const resetForm = () => {
    setFormData({
      title: "",
      host: "",
      day_of_week: "",
      start_time: "",
      end_time: "",
      description: "",
      active: true,
    });
    setEditingId(null);
    setIsAdding(false);
  };

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h2 className="text-2xl font-bold">Grade de Programação</h2>
        {!isAdding && (
          <Button variant="hero" onClick={() => setIsAdding(true)}>
            <Plus className="w-4 h-4" /> Adicionar Programa
          </Button>
        )}
      </div>

      {isAdding && (
        <Card className="p-6 space-y-4">
          <div className="flex justify-between items-center mb-4">
            <h3 className="text-lg font-semibold">
              {editingId ? "Editar Programa" : "Novo Programa"}
            </h3>
            <Button variant="ghost" size="icon" onClick={resetForm}>
              <X className="w-4 h-4" />
            </Button>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium mb-2">Título *</label>
              <Input
                value={formData.title}
                onChange={(e) => setFormData({ ...formData, title: e.target.value })}
                placeholder="Nome do programa"
              />
            </div>

            <div>
              <label className="block text-sm font-medium mb-2">Apresentador</label>
              <Input
                value={formData.host}
                onChange={(e) => setFormData({ ...formData, host: e.target.value })}
                placeholder="Nome do apresentador"
              />
            </div>

            <div>
              <label className="block text-sm font-medium mb-2">Dia da Semana *</label>
              <Select
                value={formData.day_of_week}
                onValueChange={(value) => setFormData({ ...formData, day_of_week: value })}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Selecione o dia" />
                </SelectTrigger>
                <SelectContent>
                  {DAYS_OF_WEEK.map((day) => (
                    <SelectItem key={day} value={day}>
                      {day}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="grid grid-cols-2 gap-2">
              <div>
                <label className="block text-sm font-medium mb-2">Início *</label>
                <Input
                  type="time"
                  value={formData.start_time}
                  onChange={(e) => setFormData({ ...formData, start_time: e.target.value })}
                />
              </div>
              <div>
                <label className="block text-sm font-medium mb-2">Fim *</label>
                <Input
                  type="time"
                  value={formData.end_time}
                  onChange={(e) => setFormData({ ...formData, end_time: e.target.value })}
                />
              </div>
            </div>

            <div className="md:col-span-2">
              <label className="block text-sm font-medium mb-2">Descrição</label>
              <Textarea
                value={formData.description}
                onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                placeholder="Descrição do programa..."
                rows={3}
              />
            </div>
          </div>

          <div className="flex gap-2 justify-end">
            <Button variant="outline" onClick={resetForm}>
              Cancelar
            </Button>
            <Button variant="hero" onClick={handleSave}>
              <Save className="w-4 h-4" /> Salvar
            </Button>
          </div>
        </Card>
      )}

      <div className="grid gap-4">
        {schedules.map((schedule) => (
          <Card key={schedule.id} className="p-4">
            <div className="flex justify-between items-start">
              <div className="flex-1">
                <h3 className="font-semibold text-lg">{schedule.title}</h3>
                {schedule.host && <p className="text-sm text-muted-foreground">Com: {schedule.host}</p>}
                <div className="flex gap-4 mt-2 text-sm">
                  <span className="font-medium">{schedule.day_of_week}</span>
                  <span>
                    {schedule.start_time.slice(0, 5)} - {schedule.end_time.slice(0, 5)}
                  </span>
                </div>
                {schedule.description && (
                  <p className="text-sm text-muted-foreground mt-2">{schedule.description}</p>
                )}
              </div>
              <div className="flex gap-2">
                <Button variant="outline" size="icon" onClick={() => handleEdit(schedule)}>
                  <Edit2 className="w-4 h-4" />
                </Button>
                <Button
                  variant="outline"
                  size="icon"
                  onClick={() => handleDelete(schedule.id)}
                  className="text-destructive hover:text-destructive"
                >
                  <Trash2 className="w-4 h-4" />
                </Button>
              </div>
            </div>
          </Card>
        ))}
        {schedules.length === 0 && !isAdding && (
          <Card className="p-8 text-center text-muted-foreground">
            Nenhum programa cadastrado. Clique em "Adicionar Programa" para começar.
          </Card>
        )}
      </div>
    </div>
  );
};

export default ScheduleManager;
