import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card } from "@/components/ui/card";
import { useToast } from "@/hooks/use-toast";
import { Trash2, Plus, Upload, Eye, EyeOff, ArrowUp, ArrowDown } from "lucide-react";
import { Switch } from "@/components/ui/switch";

interface Banner {
  id: string;
  image_url: string;
  title: string | null;
  link: string | null;
  active: boolean;
  position: number;
}

const BannersManager = () => {
  const { toast } = useToast();
  const [banners, setBanners] = useState<Banner[]>([]);
  const [newBanner, setNewBanner] = useState({
    image_url: "",
    title: "",
    link: "",
  });

  useEffect(() => {
    loadBanners();
  }, []);

  const loadBanners = async () => {
    const { data } = await supabase
      .from("banners")
      .select("*")
      .order("position", { ascending: true });
    
    if (data) {
      setBanners(data);
    }
  };

  const handleAddBanner = async () => {
    if (!newBanner.image_url) {
      toast({
        title: "Erro",
        description: "URL da imagem é obrigatória",
        variant: "destructive",
      });
      return;
    }

    const maxPosition = banners.length > 0 ? Math.max(...banners.map(b => b.position)) : -1;

    const { error } = await supabase.from("banners").insert({
      image_url: newBanner.image_url,
      title: newBanner.title || null,
      link: newBanner.link || null,
      position: maxPosition + 1,
      active: true,
    });

    if (error) {
      toast({
        title: "Erro ao adicionar banner",
        description: error.message,
        variant: "destructive",
      });
    } else {
      toast({ title: "Banner adicionado com sucesso!" });
      setNewBanner({ image_url: "", title: "", link: "" });
      loadBanners();
    }
  };

  const handleDeleteBanner = async (id: string) => {
    const { error } = await supabase.from("banners").delete().eq("id", id);

    if (error) {
      toast({
        title: "Erro ao remover banner",
        description: error.message,
        variant: "destructive",
      });
    } else {
      toast({ title: "Banner removido com sucesso!" });
      loadBanners();
    }
  };

  const handleToggleActive = async (id: string, active: boolean) => {
    const { error } = await supabase
      .from("banners")
      .update({ active: !active })
      .eq("id", id);

    if (error) {
      toast({
        title: "Erro ao atualizar banner",
        description: error.message,
        variant: "destructive",
      });
    } else {
      loadBanners();
    }
  };

  const handleMoveUp = async (index: number) => {
    if (index === 0) return;
    
    const newBanners = [...banners];
    const temp = newBanners[index];
    newBanners[index] = newBanners[index - 1];
    newBanners[index - 1] = temp;

    await updatePositions(newBanners);
  };

  const handleMoveDown = async (index: number) => {
    if (index === banners.length - 1) return;
    
    const newBanners = [...banners];
    const temp = newBanners[index];
    newBanners[index] = newBanners[index + 1];
    newBanners[index + 1] = temp;

    await updatePositions(newBanners);
  };

  const updatePositions = async (newBanners: Banner[]) => {
    const updates = newBanners.map((banner, index) => ({
      id: banner.id,
      position: index,
    }));

    for (const update of updates) {
      await supabase
        .from("banners")
        .update({ position: update.position })
        .eq("id", update.id);
    }

    loadBanners();
  };

  return (
    <Card className="p-6 space-y-6">
      <div>
        <h2 className="text-2xl font-bold mb-4">Gerenciar Banners (Propagandas)</h2>
        <p className="text-sm text-muted-foreground mb-6">
          Adicione até 5 banners rotativos que aparecerão no topo da página. Tamanho recomendado: 400x80 pixels
        </p>
      </div>

      {/* Adicionar Novo Banner */}
      <div className="space-y-4 p-4 border rounded-lg bg-accent/5">
        <h3 className="font-semibold flex items-center gap-2">
          <Plus className="w-4 h-4" />
          Adicionar Novo Banner
        </h3>
        <div className="space-y-3">
          <div>
            <label className="block text-sm font-medium mb-1">URL da Imagem *</label>
            <Input
              value={newBanner.image_url}
              onChange={(e) => setNewBanner({ ...newBanner, image_url: e.target.value })}
              placeholder="https://exemplo.com/banner.jpg"
            />
          </div>
          <div>
            <label className="block text-sm font-medium mb-1">Título (opcional)</label>
            <Input
              value={newBanner.title}
              onChange={(e) => setNewBanner({ ...newBanner, title: e.target.value })}
              placeholder="Nome do anúncio"
            />
          </div>
          <div>
            <label className="block text-sm font-medium mb-1">Link (opcional)</label>
            <Input
              value={newBanner.link}
              onChange={(e) => setNewBanner({ ...newBanner, link: e.target.value })}
              placeholder="https://exemplo.com"
            />
          </div>
          <Button onClick={handleAddBanner} className="w-full">
            <Upload className="w-4 h-4 mr-2" />
            Adicionar Banner
          </Button>
        </div>
      </div>

      {/* Lista de Banners */}
      <div className="space-y-3">
        <h3 className="font-semibold">Banners Cadastrados ({banners.length}/5)</h3>
        {banners.length === 0 ? (
          <p className="text-sm text-muted-foreground">Nenhum banner cadastrado ainda.</p>
        ) : (
          banners.map((banner, index) => (
            <Card key={banner.id} className="p-4">
              <div className="flex items-center gap-4">
                {/* Preview */}
                <div className="w-32 h-16 rounded overflow-hidden bg-muted flex-shrink-0">
                  <img
                    src={banner.image_url}
                    alt={banner.title || "Banner"}
                    className="w-full h-full object-cover"
                  />
                </div>

                {/* Info */}
                <div className="flex-1 min-w-0">
                  <p className="font-medium truncate">{banner.title || "Sem título"}</p>
                  {banner.link && (
                    <p className="text-xs text-muted-foreground truncate">{banner.link}</p>
                  )}
                </div>

                {/* Controles */}
                <div className="flex items-center gap-2">
                  {/* Status */}
                  <div className="flex items-center gap-2">
                    {banner.active ? (
                      <Eye className="w-4 h-4 text-green-600" />
                    ) : (
                      <EyeOff className="w-4 h-4 text-muted-foreground" />
                    )}
                    <Switch
                      checked={banner.active}
                      onCheckedChange={() => handleToggleActive(banner.id, banner.active)}
                    />
                  </div>

                  {/* Ordem */}
                  <div className="flex flex-col gap-1">
                    <Button
                      variant="ghost"
                      size="icon"
                      className="h-6 w-6"
                      onClick={() => handleMoveUp(index)}
                      disabled={index === 0}
                    >
                      <ArrowUp className="w-3 h-3" />
                    </Button>
                    <Button
                      variant="ghost"
                      size="icon"
                      className="h-6 w-6"
                      onClick={() => handleMoveDown(index)}
                      disabled={index === banners.length - 1}
                    >
                      <ArrowDown className="w-3 h-3" />
                    </Button>
                  </div>

                  {/* Deletar */}
                  <Button
                    variant="ghost"
                    size="icon"
                    onClick={() => handleDeleteBanner(banner.id)}
                    className="text-destructive hover:text-destructive"
                  >
                    <Trash2 className="w-4 h-4" />
                  </Button>
                </div>
              </div>
            </Card>
          ))
        )}
      </div>
    </Card>
  );
};

export default BannersManager;
