import { useState } from "react";
import { Card } from "./ui/card";
import { Input } from "./ui/input";
import { Button } from "./ui/button";
import { Textarea } from "./ui/textarea";
import { X, Send } from "lucide-react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

interface TestimonialChatProps {
  onClose: () => void;
}

const TestimonialChat = ({ onClose }: TestimonialChatProps) => {
  const [name, setName] = useState("");
  const [message, setMessage] = useState("");
  const [isSubmitting, setIsSubmitting] = useState(false);
  const { toast } = useToast();

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!name.trim() || !message.trim()) {
      toast({
        title: "Campos obrigatórios",
        description: "Por favor, preencha seu nome e mensagem",
        variant: "destructive",
      });
      return;
    }

    setIsSubmitting(true);
    
    const { error } = await supabase.from("testimonials").insert({
      name: name.trim(),
      message: message.trim(),
    });

    setIsSubmitting(false);

    if (error) {
      toast({
        title: "Erro ao enviar",
        description: "Não foi possível enviar seu testemunho. Tente novamente.",
        variant: "destructive",
      });
    } else {
      toast({
        title: "Testemunho enviado!",
        description: "Obrigado por compartilhar sua história conosco.",
      });
      setName("");
      setMessage("");
      setTimeout(onClose, 2000);
    }
  };

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center p-4 bg-background/80 backdrop-blur-sm">
      <Card className="w-full max-w-lg p-6 space-y-4 animate-in fade-in-50 slide-in-from-bottom-10">
        <div className="flex items-center justify-between">
          <h3 className="text-2xl font-bold bg-gradient-to-r from-primary to-[hsl(var(--primary-variant))] bg-clip-text text-transparent">
            Compartilhe Seu Testemunho
          </h3>
          <Button variant="ghost" size="icon" onClick={onClose}>
            <X className="w-5 h-5" />
          </Button>
        </div>

        <form onSubmit={handleSubmit} className="space-y-4">
          <div>
            <label className="block text-sm font-medium mb-2">Seu Nome</label>
            <Input
              value={name}
              onChange={(e) => setName(e.target.value)}
              placeholder="Digite seu nome"
              maxLength={100}
              required
            />
          </div>

          <div>
            <label className="block text-sm font-medium mb-2">Seu Testemunho</label>
            <Textarea
              value={message}
              onChange={(e) => setMessage(e.target.value)}
              placeholder="Compartilhe como a Rádio Mega Gospel impactou sua vida..."
              className="min-h-[150px] resize-none"
              maxLength={1000}
              required
            />
            <p className="text-xs text-muted-foreground mt-1">
              {message.length}/1000 caracteres
            </p>
          </div>

          <Button
            type="submit"
            variant="hero"
            className="w-full"
            disabled={isSubmitting}
          >
            <Send className="w-4 h-4 mr-2" />
            {isSubmitting ? "Enviando..." : "Enviar Testemunho"}
          </Button>
        </form>
      </Card>
    </div>
  );
};

export default TestimonialChat;
