import { useState, useRef, useEffect } from "react";
import { Play, Pause, Volume2, VolumeX } from "lucide-react";
import { Button } from "./ui/button";
import { Slider } from "./ui/slider";
import { supabase } from "@/integrations/supabase/client";
import { toast } from "@/components/ui/use-toast";
import BannersCarousel from "./BannersCarousel";

const DEFAULT_STREAM = "https://stm.painelmegahost.com.br:7004/stream";
const MediaPlayer = () => {
  const [isPlaying, setIsPlaying] = useState(false);
  const [volume, setVolume] = useState(70);
  const [isMuted, setIsMuted] = useState(false);
  const [currentShow, setCurrentShow] = useState("Programação Gospel");
  const [streamUrl, setStreamUrl] = useState(DEFAULT_STREAM);
  const audioRef = useRef<HTMLAudioElement>(null);

  useEffect(() => {
    loadSettings();
  }, []);

  const loadSettings = async () => {
    const { data } = await supabase.from("site_settings").select("*");
    if (data) {
      const settingsObj = data.reduce((acc, item) => ({ ...acc, [item.key]: item.value }), {} as Record<string, string>);
      // Use a URL padrão se não estiver configurada
      setStreamUrl(settingsObj.radio_stream_url || DEFAULT_STREAM);
      setCurrentShow(settingsObj.current_show || "Programação Gospel");
    }
  };

  useEffect(() => {
    if (audioRef.current) {
      audioRef.current.volume = volume / 100;
    }
  }, [volume]);

  const togglePlay = async () => {
    if (!audioRef.current || !streamUrl) {
      console.error("Stream URL não configurado");
      return;
    }
    
    try {
      if (isPlaying) {
        audioRef.current.pause();
        setIsPlaying(false);
      } else {
        // Tenta diferentes formatos/endereços do servidor (Shoutcast)
        const candidates = Array.from(
          new Set([
            streamUrl,
            streamUrl.replace(/\/stream$/, "/;stream.mp3"),
            streamUrl.replace(/\/stream$/, "/;"),
            DEFAULT_STREAM,
            "https://stm.painelmegahost.com.br:7004/;stream.mp3",
            "https://stm.painelmegahost.com.br:7004/;",
          ])
        );

        let success = false;
        for (const url of candidates) {
          try {
            if (audioRef.current.src !== url) {
              audioRef.current.src = url;
            }
            audioRef.current.load();
            await audioRef.current.play();
            setIsPlaying(true);
            if (url !== streamUrl) setStreamUrl(url);
            success = true;
            break;
          } catch (err) {
            console.warn("Falha ao tentar URL:", url, err);
            continue;
          }
        }

        if (!success) {
          throw new Error("Falha em todas as tentativas de stream");
        }
      }
    } catch (error: any) {
      console.error("Erro ao tocar áudio:", error);
      if (error?.name === 'NotAllowedError') {
        console.warn("Reprodução bloqueada pelo navegador - requer interação do usuário");
      }
      toast({
        title: "Falha ao reproduzir",
        description: "Não foi possível iniciar o stream agora. Tente novamente.",
        variant: "destructive",
      });
      setIsPlaying(false);
    }
  };

  const toggleMute = () => {
    if (audioRef.current) {
      audioRef.current.muted = !isMuted;
      setIsMuted(!isMuted);
    }
  };

  const handleVolumeChange = (value: number[]) => {
    setVolume(value[0]);
    if (audioRef.current) {
      audioRef.current.volume = value[0] / 100;
    }
  };

  return (
    <div className="sticky top-20 z-40 bg-gradient-to-r from-primary via-[hsl(var(--primary-variant))] to-primary text-primary-foreground shadow-lg">
      <div className="container mx-auto px-4 py-4">
        <div className="flex flex-col md:flex-row items-center justify-between gap-4">
          {/* Info do Programa */}
          <div className="flex items-center gap-3 flex-1">
            <div className="w-12 h-12 rounded-full bg-secondary flex items-center justify-center animate-pulse">
              <div className="w-8 h-8 rounded-full bg-accent"></div>
            </div>
            <div>
              <p className="text-xs font-medium opacity-90">NO AR AGORA</p>
              <p className="font-bold text-lg">{currentShow}</p>
            </div>
          </div>

          {/* Banners Carousel */}
          <BannersCarousel />

          {/* Controles */}
          <div className="flex items-center gap-4">
            <Button
              variant="ghost"
              size="icon"
              onClick={togglePlay}
              className="w-12 h-12 rounded-full bg-white/20 hover:bg-white/30 text-white"
            >
              {isPlaying ? <Pause className="w-6 h-6" /> : <Play className="w-6 h-6 ml-1" />}
            </Button>

            {/* Volume */}
            <div className="hidden md:flex items-center gap-2 w-32">
              <button onClick={toggleMute} className="text-white hover:opacity-80 transition-opacity">
                {isMuted ? <VolumeX className="w-5 h-5" /> : <Volume2 className="w-5 h-5" />}
              </button>
              <Slider
                value={[volume]}
                onValueChange={handleVolumeChange}
                max={100}
                step={1}
                className="w-full"
              />
            </div>

            <div className="hidden lg:flex items-center gap-2 px-4 py-2 rounded-full bg-white/10">
              <div className="w-2 h-2 rounded-full bg-accent animate-pulse"></div>
              <span className="text-xs font-medium">AO VIVO</span>
            </div>
          </div>
        </div>
      </div>

      {/* Audio Element */}
      <audio 
        ref={audioRef}
        preload="none"
        onError={(e) => {
          const target = e.currentTarget as HTMLAudioElement;
          const code = target.error?.code;
          console.error("Erro no áudio:", target.error);
          toast({ title: "Erro no stream", description: `Código: ${code ?? "desconhecido"}` , variant: "destructive"});
          setIsPlaying(false);
        }}
        onCanPlay={() => console.log("Áudio pronto para tocar")}
      >
        <source src={streamUrl} type="audio/mp3" />
        <source src={streamUrl} type="audio/mpeg" />
      </audio>
    </div>
  );
};

export default MediaPlayer;
