import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import {
  Carousel,
  CarouselContent,
  CarouselItem,
  CarouselNext,
  CarouselPrevious,
} from "@/components/ui/carousel";
import Autoplay from "embla-carousel-autoplay";

interface Banner {
  id: string;
  image_url: string;
  title: string | null;
  link: string | null;
  active: boolean;
  position: number;
}

const BannersCarousel = () => {
  const [banners, setBanners] = useState<Banner[]>([]);

  useEffect(() => {
    loadBanners();
  }, []);

  const loadBanners = async () => {
    const { data } = await supabase
      .from("banners")
      .select("*")
      .eq("active", true)
      .order("position", { ascending: true });
    
    if (data && data.length > 0) {
      setBanners(data);
    }
  };

  if (banners.length === 0) return null;

  const handleBannerClick = (link: string | null) => {
    if (link) {
      window.open(link, "_blank", "noopener,noreferrer");
    }
  };

  return (
    <div className="w-full max-w-xs">
      <Carousel
        opts={{
          align: "start",
          loop: true,
        }}
        plugins={[
          Autoplay({
            delay: 4000,
          }),
        ]}
        className="w-full"
      >
        <CarouselContent>
          {banners.map((banner) => (
            <CarouselItem key={banner.id}>
              <div
                className={`relative h-16 rounded-lg overflow-hidden ${
                  banner.link ? "cursor-pointer" : ""
                }`}
                onClick={() => handleBannerClick(banner.link)}
              >
                <img
                  src={banner.image_url}
                  alt={banner.title || "Banner"}
                  className="w-full h-full object-cover"
                />
              </div>
            </CarouselItem>
          ))}
        </CarouselContent>
        <CarouselPrevious className="left-1 h-6 w-6" />
        <CarouselNext className="right-1 h-6 w-6" />
      </Carousel>
    </div>
  );
};

export default BannersCarousel;
